#!/opt/perl-5.12.2-emg/bin/perl
# Script hourly_summary_connector.pl
#
# Aggregates data from routelog to emgp_hourly_summary_connector for fast generation of reports.
# Script should be run every minute or so by cron.
# It will store the last message timestamp processed in table "emgsystem", keyname "hourly_summary_connector_next_ts"
# A MySQL lock need to be acquired during execution in order to prevent multiple simultaneous runs.

use strict;

use DBI;
use DBI qw(:sql_types);
use File::Basename;
use Time::HiRes qw(gettimeofday usleep);

# These should be overridden in the hourly_summary.cfg file.
our $dbname = 'emg';
our $dbuser = 'emguser';
our $dbpasswd = 'secret';
our $dbhost = '127.0.0.1';

# *** No changes should be necessary below this line ***

my $dirname = dirname(__FILE__);

# MySQL lock name
my $lock_name = 'hourly_summary_connector_run';
# Configuration file where we can override above "our" variables
my $cfg_file = "$dirname/hourly_summary.cfg";

if(-f $cfg_file) {
	require $cfg_file;
}

my $dbh;
my $sth_select_next_ts;
my $sth_select_min_starttime;
my $sth_insert_next_ts;
my $sth_update_next_ts;
my $sth_get_next_ts;
my $sth_select_relayed;
my $sth_clear_msg_count;
my $sth_select_new;
my $sth_insert_or_update_new;

sub format_time
{
	my $seconds = shift;
	my $msecs = shift;

	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($seconds);

	if(!defined($msecs)) {
		sprintf "%04d-%02d-%02d %02d:%02d:%02d", $year + 1900, $mon + 1, $mday, $hour, $min, $sec;
	} else {
		sprintf "%04d-%02d-%02d %02d:%02d:%02d.%03d", $year + 1900, $mon + 1, $mday, $hour, $min, $sec, $msecs;
	}
}

sub log_with_time
{
	my $text = shift;
	my ($seconds, $microseconds) = gettimeofday;
	my $msecs = $microseconds / 1000;

	my $formatted_time = format_time($seconds, $msecs);

	printf "%s - %s\n", $formatted_time, $text;
}

sub db_connect
{
	$dbh = DBI->connect("DBI:mysql:$dbname:$dbhost", "$dbuser", "$dbpasswd", { AutoCommit => 0 });
	# Get "next_ts"
	$sth_select_next_ts = $dbh->prepare(qq{
		SELECT `value` FROM emgsystem WHERE `keyname` = "hourly_summary_connector_next_ts"
		});
	# Get least value for starttime
	$sth_select_min_starttime = $dbh->prepare(qq{
		SELECT LEFT(MIN(starttime), 13) FROM routelog
		});
	# Insert "next_ts"
	$sth_insert_next_ts = $dbh->prepare(qq{
		INSERT INTO emgsystem (`keyname`, `value`) VALUES("hourly_summary_connector_next_ts", ?)
		});
	# Update "next_ts"
	$sth_update_next_ts = $dbh->prepare(qq{
		UPDATE emgsystem SET `value` = ? WHERE `keyname` = "hourly_summary_connector_next_ts"
		});
	# Get next hour
	$sth_get_next_ts = $dbh->prepare(qq{
		SELECT LEFT(DATE_ADD(?, INTERVAL 1 HOUR), 13) FROM emgsystem WHERE ? < LEFT(NOW(), 13) LIMIT 1
		});
	# Get entries in specific time range (now - 3 days) which have "relayed" count
	$sth_select_relayed = $dbh->prepare(qq{
		SELECT time, SUM(msg_count_relayed) FROM emgp_hourly_summary_connector
			WHERE time >= ? AND time <= ? AND msg_count_relayed > 0
			GROUP BY 1
		});
	# Reset ok and relayed counters
	$sth_clear_msg_count = $dbh->prepare(qq{
		DELETE FROM emgp_hourly_summary_connector
			WHERE time = ?
		});
	# Get new entries from routelog
	$sth_select_new = $dbh->prepare(qq{
		SELECT username,
			   inconnector,
			   outconnector,
			   r.status,
			   SUM(IFNULL(npdus, 0)),
			   charge,
			   charge_cost
			FROM routelog r
			WHERE starttime >= ? AND
				  starttime < DATE_ADD(?, INTERVAL 1 HOUR) AND
			  	  msgtype = 1 AND
				  inconnector IS NOT NULL AND
				  outconnector IS NOT NULL
			GROUP BY 1, 2, 3, 4, 6, 7
		});
	# Insert (or update) hourly summaries with new entries
	$sth_insert_or_update_new = $dbh->prepare(qq{
		INSERT INTO emgp_hourly_summary_connector
			(username, time, connector_in, connector_out, charge, charge_cost, msg_count, msg_count_ok, msg_count_relayed)
			VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?)
		});
}

sub acquire_lock
{
	my $sth = $dbh->prepare("SELECT GET_LOCK('$lock_name', 0)");
	$sth->execute;
	my ($lock) = $sth->fetchrow;
	return $lock;
}

sub release_lock
{
	$dbh->do("SELECT RELEASE_LOCK('$lock_name')");
}

sub get_next_ts
{
	$sth_select_next_ts->execute;
	my @data = $sth_select_next_ts->fetchrow_array;
	if(!$data[0]) {
		$sth_select_min_starttime->execute;
		@data = $sth_select_min_starttime->fetchrow_array;
		if($data[0]) {
			$sth_insert_next_ts->execute($data[0]);
			$dbh->commit;
		}
	}
	my $next_ts = $data[0] || '2010-01-01 00';
	#print "next_ts=$next_ts\n";
	return $next_ts;
}

sub inc_next_ts
{
	my $next_ts = get_next_ts;
	$sth_get_next_ts->execute($next_ts, $next_ts);
	my @data = $sth_get_next_ts->fetchrow_array;
	if($data[0]) {
		$next_ts = $data[0];
		$sth_update_next_ts->execute($next_ts);
		$dbh->commit;
	}
	$next_ts;
}

sub insert_new_entries
{
	my $next_ts = shift;

	while(1) {
		calc_for_ts($next_ts);
		my $old_ts = $next_ts;
		$next_ts = inc_next_ts;
		if($old_ts eq $next_ts) {
			last;
		}
	}
}

sub calc_for_ts
{
	my $ts = shift;

	# Clear any current data
	$sth_clear_msg_count->execute($ts);

	log_with_time "Collecting entries for hour $ts";
	$sth_select_new->execute($ts, $ts);

	my $counters;
	while(my @data = $sth_select_new->fetchrow_array) {
		my $username = $data[0];
		my $connector_in = $data[1];
		my $connector_out = $data[2];
		my $status = $data[3];
		my $npdus = $data[4] || 1;
		my $charge = $data[5] || 0;
		my $charge_cost = $data[6] || 0;

		my $npdus_ok = 0;
		my $npdus_relayed = 0;

		$npdus_ok = $npdus if($status == 1);
		$npdus_relayed = $npdus if($status == 10 || $status == 8);

		#(username, time, connector_in, connector_out, charge, charge_cost, msg_count, msg_count_ok, msg_count_relayed)
		my $key = "$username:$ts:$connector_in:$connector_out";
		if(!defined($counters->{$key})) {
			$counters->{$key}{'username'} = $username;
			$counters->{$key}{'time'} = $ts;
			$counters->{$key}{'connector_in'} = $connector_in;
			$counters->{$key}{'connector_out'} = $connector_out;
			$counters->{$key}{'charge'} = 0;
			$counters->{$key}{'charge_cost'} = 0;
			$counters->{$key}{'npdus'} = 0;
			$counters->{$key}{'npdus_ok'} = 0;
			$counters->{$key}{'npdus_relayed'} = 0;
		}
		$counters->{$key}{'charge'} += $npdus * $charge;
		$counters->{$key}{'charge_cost'} += $npdus * $charge_cost;
		$counters->{$key}{'npdus'} += $npdus;
		$counters->{$key}{'npdus_ok'} += $npdus_ok;
		$counters->{$key}{'npdus_relayed'} += $npdus_relayed;
	}
	foreach my $key (keys %$counters) {
		$sth_insert_or_update_new->bind_param(1, $counters->{$key}{'username'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(2, $counters->{$key}{'time'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(3, $counters->{$key}{'connector_in'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(4, $counters->{$key}{'connector_out'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(5, $counters->{$key}{'charge'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(6, $counters->{$key}{'charge_cost'}, SQL_VARCHAR);
		$sth_insert_or_update_new->bind_param(7, $counters->{$key}{'npdus'}, SQL_INTEGER);
		$sth_insert_or_update_new->bind_param(8, $counters->{$key}{'npdus_ok'}, SQL_INTEGER);
		$sth_insert_or_update_new->bind_param(9, $counters->{$key}{'npdus_relayed'}, SQL_INTEGER);
		$sth_insert_or_update_new->execute;
	}
	$dbh->commit;
}

sub update_relayed_entries
{
	my $next_ts = shift;
	# We go 3 days back
	my $from = substr(format_time(time - 3 * 24 * 3600), 0, 13);
	my $to = substr($next_ts, 0, 13);
	log_with_time "Check from $from to $to";
	$sth_select_relayed->execute($from, $to);
	while(my @data = $sth_select_relayed->fetchrow_array) {
		my $time = $data[0];
		my $msg_count_relayed = $data[1];
		log_with_time "Found time $time, relayed messages $msg_count_relayed";
		calc_for_ts($time);
	}
}

log_with_time "Generating hourly summary";

if (!db_connect) {
	log_with_time "Could not open database, is $cfg_file correct?";
	exit 1;
}

if(!acquire_lock) {
	log_with_time "Could not acquire run lock";
	exit 1;
}
my $next_ts = get_next_ts;

log_with_time "Will check for existing, relayed entries to update";
update_relayed_entries($next_ts);

log_with_time "Will check for new entries";
insert_new_entries($next_ts);

log_with_time "Completed";
release_lock;

